#!/bin/bash

# 颜色定义
BOLD='\033[1m'
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[0;33m'
BLUE='\033[0;34m'
MAGENTA='\033[0;35m'
CYAN='\033[0;36m'
WHITE='\033[0;37m'
RESET='\033[0m'

# 简单的渐变色实现（替代 lolcat）
rainbow_text() {
    local text="$1"
    local colors=("$RED" "$YELLOW" "$GREEN" "$CYAN" "$BLUE" "$MAGENTA")
    local color_count=${#colors[@]}
    local line_num=0
    
    while IFS= read -r line; do
        local color_index=$((line_num % color_count))
        echo -e "${colors[$color_index]}${line}${RESET}"
        ((line_num++))
    done <<< "$text"
}

# ASCII Logo - 已禁用
print_logo() {
    :  # 空操作
}

# 获取系统信息
get_os() {
    if [ -f /etc/os-release ]; then
        . /etc/os-release
        echo "$NAME $VERSION_ID"
    elif [ -f /etc/lsb-release ]; then
        . /etc/lsb-release
        echo "$DISTRIB_DESCRIPTION"
    else
        uname -s
    fi
}

get_kernel() {
    uname -r
}

get_uptime() {
    if [ -f /proc/uptime ]; then
        uptime_seconds=$(cat /proc/uptime | awk '{print int($1)}')
        days=$((uptime_seconds / 86400))
        hours=$(((uptime_seconds % 86400) / 3600))
        mins=$(((uptime_seconds % 3600) / 60))
        
        result=""
        [ $days -gt 0 ] && result="${days}天 "
        [ $hours -gt 0 ] && result="${result}${hours}小时 "
        result="${result}${mins}分钟"
        echo "$result"
    else
        uptime | awk -F'up ' '{print $2}' | awk -F',' '{print $1}'
    fi
}

get_packages() {
    local count=0
    if command -v dpkg &> /dev/null; then
        count=$(dpkg -l | grep -c '^ii')
        echo "$count (dpkg)"
    elif command -v rpm &> /dev/null; then
        count=$(rpm -qa | wc -l)
        echo "$count (rpm)"
    elif command -v pacman &> /dev/null; then
        count=$(pacman -Q | wc -l)
        echo "$count (pacman)"
    elif command -v apk &> /dev/null; then
        count=$(apk list --installed 2>/dev/null | wc -l)
        echo "$count (apk)"
    else
        echo "未知"
    fi
}

get_shell() {
    basename "$SHELL"
}

get_resolution() {
    if command -v xrandr &> /dev/null; then
        xrandr | grep '*' | awk '{print $1}' | head -n1
    else
        echo "未知"
    fi
}

get_de() {
    echo "${XDG_CURRENT_DESKTOP:-${DESKTOP_SESSION:-未知}}"
}

get_wm() {
    if [ -n "$WAYLAND_DISPLAY" ]; then
        echo "Wayland"
    elif [ -n "$DISPLAY" ]; then
        echo "X11"
    else
        echo "TTY"
    fi
}

get_terminal() {
    # 兼容 BusyBox ps
    if ps -o comm 2>/dev/null | head -n1 | grep -q "COMMAND"; then
        ps -o comm | grep -v "COMMAND" | head -n1
    else
        echo "${TERM:-unknown}"
    fi
}

get_cpu() {
    if [ -f /proc/cpuinfo ]; then
        grep -m1 "model name" /proc/cpuinfo | cut -d: -f2 | sed 's/^[ \t]*//'
    else
        uname -m
    fi
}

get_gpu() {
    if command -v lspci &> /dev/null; then
        lspci | grep -i vga | cut -d: -f3 | sed 's/^[ \t]*//'
    else
        echo "未知"
    fi
}

get_memory() {
    if [ -f /proc/meminfo ]; then
        total=$(grep MemTotal /proc/meminfo | awk '{print $2}')
        available=$(grep MemAvailable /proc/meminfo | awk '{print $2}')
        used=$((total - available))
        
        used_mb=$((used / 1024))
        total_mb=$((total / 1024))
        
        echo "${used_mb}MB / ${total_mb}MB"
    else
        echo "未知"
    fi
}

get_disk() {
    df -h / | awk 'NR==2 {print $3 " / " $2 " (" $5 ")"}'
}

# 打印信息行
print_info() {
    local label="$1"
    local value="$2"
    local color="$3"
    printf "${BOLD}${color}%-12s${RESET} %s\n" "$label" "$value"
}

# 主函数
main() {
    clear
    
    # 获取用户名和主机名
    USER_HOST="${USER}@${HOSTNAME}"
    
    # 打印 logo (已禁用)
    # print_logo
    
    # 打印开发者信息 - 使用 figlet 和自制渐变色
   figlet -f slant -w 45 -c "Yuwan X" | lolcat
	figlet -f slant -w 90 -c "Uzuma.ru" | lolcat

    # 打印系统信息
    echo -e "${BOLD}${GREEN}${USER_HOST}${RESET}"
    printf '%*s\n' "${#USER_HOST}" | tr ' ' '-'
    echo
    
    print_info "OS:" "$(get_os)" "$BLUE"
    print_info "Kernel:" "$(get_kernel)" "$BLUE"
    print_info "Uptime:" "$(get_uptime)" "$BLUE"
    print_info "Packages:" "$(get_packages)" "$BLUE"
    print_info "Shell:" "$(get_shell)" "$BLUE"
    print_info "WM:" "$(get_wm)" "$BLUE"
    print_info "Terminal:" "$(get_terminal)" "$BLUE"
    print_info "CPU:" "$(get_cpu)" "$BLUE"
    print_info "Memory:" "$(get_memory)" "$BLUE"
    print_info "Disk (/):" "$(get_disk)" "$BLUE"
    
    echo
    
    # 颜色条
    echo -e "${BLACK}███${RED}███${GREEN}███${YELLOW}███${BLUE}███${MAGENTA}███${CYAN}███${WHITE}███${RESET}"
    echo
        # 欢迎信息
    echo -e "欢迎使用Alpine 3.19 Uzmaru 特调版"
    echo -e "---------------------------------------"
    echo -e "您可以使用一键命令 ${GREEN}alpine${RESET} 来做你需要的事"
    echo -e "-----------------------------------------------------------------"
    echo -e "crontab中包含了定时三天删除/var/log下的journalctl syslog等日志文件"
    echo -e "您可以按需删除定时任务 其中您可以使用一键命令 ${GREEN}alpine${RESET} 来做你需要的事"

    echo

       # 广告信息
    echo -e "${WHITE}=======广=======告=======时=======间=======${RESET}"
    echo -e "${RED}Uzumaru 专业团队${RESET} ${BLUE}>>${RESET} ${YELLOW}https://uzuma.ru${RESET} ${BLUE}<<${RESET}"
    echo -e "${GREEN}提供全球多地区NAT VPS${RESET}"
    echo
}

# 运行
main
